"use strict";
/*
 * Copyright 2020 gRPC authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.XdsClient = void 0;
const protoLoader = require("@grpc/proto-loader");
const make_client_1 = require("./make-client");
const channel_credentials_1 = require("./channel-credentials");
const xds_bootstrap_1 = require("./xds-bootstrap");
const net_1 = require("net");
const constants_1 = require("./constants");
const metadata_1 = require("./metadata");
const logging = require("./logging");
const TRACER_NAME = 'xds_client';
function trace(text) {
    logging.trace(constants_1.LogVerbosity.DEBUG, TRACER_NAME, text);
}
const clientVersion = require('../../package.json').version;
const EDS_TYPE_URL = 'type.googleapis.com/envoy.api.v2.ClusterLoadAssignment';
let loadedProtos = null;
function loadAdsProtos() {
    if (loadedProtos !== null) {
        return loadedProtos;
    }
    loadedProtos = protoLoader
        .load([
        'envoy/service/discovery/v2/ads.proto',
        'envoy/api/v2/listener.proto',
        'envoy/api/v2/route.proto',
        'envoy/api/v2/cluster.proto',
        'envoy/api/v2/endpoint.proto',
    ], {
        keepCase: true,
        longs: String,
        enums: String,
        defaults: true,
        oneofs: true,
        includeDirs: [
            // Paths are relative to src/build
            __dirname + '/../../deps/envoy-api/',
            __dirname + '/../../deps/udpa/',
            __dirname + '/../../deps/googleapis/',
            __dirname + '/../../deps/protoc-gen-validate/',
        ],
    })
        .then((packageDefinition) => make_client_1.loadPackageDefinition(packageDefinition));
    return loadedProtos;
}
class XdsClient {
    constructor(targetName, serviceConfigWatcher, channelOptions) {
        this.targetName = targetName;
        this.serviceConfigWatcher = serviceConfigWatcher;
        this.node = null;
        this.client = null;
        this.adsCall = null;
        this.hasShutdown = false;
        this.endpointWatchers = new Map();
        this.lastEdsVersionInfo = '';
        this.lastEdsNonce = '';
        this.latestEdsResponses = [];
        const channelArgs = Object.assign({}, channelOptions);
        const channelArgsToRemove = [
            /* The SSL target name override corresponds to the target, and this
             * client has its own target */
            'grpc.ssl_target_name_override',
            /* The default authority also corresponds to the target */
            'grpc.default_authority',
            /* This client will have its own specific keepalive time setting */
            'grpc.keepalive_time_ms',
            /* The service config specifies the load balancing policy. This channel
             * needs its own separate load balancing policy setting. In particular,
             * recursively using an xDS load balancer for the xDS client would be
             * bad */
            'grpc.service_config',
        ];
        for (const arg of channelArgsToRemove) {
            delete channelArgs[arg];
        }
        channelArgs['grpc.keepalive_time_ms'] = 5000;
        Promise.all([xds_bootstrap_1.loadBootstrapInfo(), loadAdsProtos()]).then(([bootstrapInfo, protoDefinitions]) => {
            if (this.hasShutdown) {
                return;
            }
            this.node = Object.assign(Object.assign({}, bootstrapInfo.node), { build_version: `gRPC Node Pure JS ${clientVersion}`, user_agent_name: 'gRPC Node Pure JS' });
            this.client = new protoDefinitions.envoy.service.discovery.v2.AggregatedDiscoveryService(bootstrapInfo.xdsServers[0].serverUri, channel_credentials_1.createGoogleDefaultCredentials(), channelArgs);
            this.maybeStartAdsStream();
        }, (error) => {
            trace('Failed to initialize xDS Client. ' + error.message);
            // Bubble this error up to any listeners
            this.reportStreamError({
                code: constants_1.Status.INTERNAL,
                details: `Failed to initialize xDS Client. ${error.message}`,
                metadata: new metadata_1.Metadata(),
            });
        });
    }
    /**
     * Start the ADS stream if the client exists and there is not already an
     * existing stream, and there
     */
    maybeStartAdsStream() {
        if (this.client === null) {
            return;
        }
        if (this.adsCall !== null) {
            return;
        }
        if (this.hasShutdown) {
            return;
        }
        this.adsCall = this.client.StreamAggregatedResources();
        this.adsCall.on('data', (message) => {
            switch (message.type_url) {
                case EDS_TYPE_URL: {
                    const edsResponses = [];
                    for (const resource of message.resources) {
                        if (protoLoader.isAnyExtension(resource) &&
                            resource['@type'] === EDS_TYPE_URL) {
                            const resp = resource;
                            if (!this.validateEdsResponse(resp)) {
                                this.nackEds('ClusterLoadAssignment validation failed');
                                return;
                            }
                            edsResponses.push(resp);
                        }
                        else {
                            this.nackEds(`Invalid resource type ${protoLoader.isAnyExtension(resource)
                                ? resource['@type']
                                : resource.type_url}`);
                            return;
                        }
                    }
                    for (const message of edsResponses) {
                        this.handleEdsResponse(message);
                    }
                    this.lastEdsVersionInfo = message.version_info;
                    this.lastEdsNonce = message.nonce;
                    this.latestEdsResponses = edsResponses;
                    this.ackEds();
                    break;
                }
                default:
                    this.nackUnknown(message.type_url, message.version_info, message.nonce);
            }
        });
        this.adsCall.on('error', (error) => {
            trace('ADS stream ended. code=' + error.code + ' details= ' + error.details);
            this.adsCall = null;
            this.reportStreamError(error);
            /* Connection backoff is handled by the client object, so we can
             * immediately start a new request to indicate that it should try to
             * reconnect */
            this.maybeStartAdsStream();
        });
        const endpointWatcherNames = Array.from(this.endpointWatchers.keys());
        if (endpointWatcherNames.length > 0) {
            this.adsCall.write({
                node: this.node,
                type_url: EDS_TYPE_URL,
                resource_names: endpointWatcherNames,
            });
        }
    }
    nackUnknown(typeUrl, versionInfo, nonce) {
        if (!this.adsCall) {
            return;
        }
        this.adsCall.write({
            node: this.node,
            type_url: typeUrl,
            version_info: versionInfo,
            response_nonce: nonce,
            error_detail: {
                message: `Unknown type_url ${typeUrl}`,
            },
        });
    }
    /**
     * Acknowledge an EDS update. This should be called after the local nonce and
     * version info are updated so that it sends the post-update values.
     */
    ackEds() {
        if (!this.adsCall) {
            return;
        }
        this.adsCall.write({
            node: this.node,
            type_url: EDS_TYPE_URL,
            resource_names: Array.from(this.endpointWatchers.keys()),
            response_nonce: this.lastEdsNonce,
            version_info: this.lastEdsVersionInfo,
        });
    }
    /**
     * Reject an EDS update. This should be called without updating the local
     * nonce and version info.
     */
    nackEds(message) {
        if (!this.adsCall) {
            return;
        }
        this.adsCall.write({
            node: this.node,
            type_url: EDS_TYPE_URL,
            resource_names: Array.from(this.endpointWatchers.keys()),
            response_nonce: this.lastEdsNonce,
            version_info: this.lastEdsVersionInfo,
            error_detail: {
                message,
            },
        });
    }
    /**
     * Validate the ClusterLoadAssignment object by these rules:
     * https://github.com/grpc/proposal/blob/master/A27-xds-global-load-balancing.md#clusterloadassignment-proto
     * @param message
     */
    validateEdsResponse(message) {
        var _a, _b;
        for (const endpoint of message.endpoints) {
            for (const lb of endpoint.lb_endpoints) {
                const socketAddress = (_b = (_a = lb.endpoint) === null || _a === void 0 ? void 0 : _a.address) === null || _b === void 0 ? void 0 : _b.socket_address;
                if (!socketAddress) {
                    return false;
                }
                if (socketAddress.port_specifier !== 'port_value') {
                    return false;
                }
                if (!(net_1.isIPv4(socketAddress.address) || net_1.isIPv6(socketAddress.address))) {
                    return false;
                }
            }
        }
        return true;
    }
    handleEdsResponse(message) {
        var _a;
        const watchers = (_a = this.endpointWatchers.get(message.cluster_name)) !== null && _a !== void 0 ? _a : [];
        for (const watcher of watchers) {
            watcher.onValidUpdate(message);
        }
    }
    updateEdsNames() {
        if (this.adsCall) {
            this.adsCall.write({
                node: this.node,
                type_url: EDS_TYPE_URL,
                resource_names: Array.from(this.endpointWatchers.keys()),
                response_nonce: this.lastEdsNonce,
                version_info: this.lastEdsVersionInfo,
            });
        }
    }
    reportStreamError(status) {
        for (const watcherList of this.endpointWatchers.values()) {
            for (const watcher of watcherList) {
                watcher.onTransientError(status);
            }
        }
        // Also do the same for other types of watchers when those are implemented
    }
    addEndpointWatcher(edsServiceName, watcher) {
        trace('Watcher added for endpoint ' + edsServiceName);
        let watchersEntry = this.endpointWatchers.get(edsServiceName);
        let addedServiceName = false;
        if (watchersEntry === undefined) {
            addedServiceName = true;
            watchersEntry = [];
            this.endpointWatchers.set(edsServiceName, watchersEntry);
        }
        watchersEntry.push(watcher);
        if (addedServiceName) {
            this.updateEdsNames();
        }
        /* If we have already received an update for the requested edsServiceName,
         * immediately pass that update along to the watcher */
        for (const message of this.latestEdsResponses) {
            if (message.cluster_name === edsServiceName) {
                /* These updates normally occur asynchronously, so we ensure that
                 * the same happens here */
                process.nextTick(() => {
                    watcher.onValidUpdate(message);
                });
            }
        }
    }
    removeEndpointWatcher(edsServiceName, watcher) {
        trace('Watcher removed for endpoint ' + edsServiceName);
        const watchersEntry = this.endpointWatchers.get(edsServiceName);
        let removedServiceName = false;
        if (watchersEntry !== undefined) {
            const entryIndex = watchersEntry.indexOf(watcher);
            if (entryIndex >= 0) {
                watchersEntry.splice(entryIndex, 1);
            }
            if (watchersEntry.length === 0) {
                removedServiceName = true;
                this.endpointWatchers.delete(edsServiceName);
            }
        }
        if (removedServiceName) {
            this.updateEdsNames();
        }
    }
    shutdown() {
        var _a, _b;
        (_a = this.adsCall) === null || _a === void 0 ? void 0 : _a.cancel();
        (_b = this.client) === null || _b === void 0 ? void 0 : _b.close();
        this.hasShutdown = true;
    }
}
exports.XdsClient = XdsClient;
//# sourceMappingURL=xds-client.js.map