#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdbool.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/wait.h>

#include "dshlib.h"

/*
 * Implement your exec_local_cmd_loop function by building a loop that prompts the 
 * user for input.  Use the SH_PROMPT constant from dshlib.h and then
 * use fgets to accept user input.
 * 
 *      while(1){
 *        printf("%s", SH_PROMPT);
 *        if (fgets(cmd_buff, ARG_MAX, stdin) == NULL){
 *           printf("\n");
 *           break;
 *        }
 *        //remove the trailing \n from cmd_buff
 *        cmd_buff[strcspn(cmd_buff,"\n")] = '\0';
 * 
 *        //IMPLEMENT THE REST OF THE REQUIREMENTS
 *      }
 * 
 *   Also, use the constants in the dshlib.h in this code.  
 *      SH_CMD_MAX              maximum buffer size for user input
 *      EXIT_CMD                constant that terminates the dsh program
 *      SH_PROMPT               the shell prompt
 *      OK                      the command was parsed properly
 *      WARN_NO_CMDS            the user command was empty
 *      ERR_TOO_MANY_COMMANDS   too many pipes used
 *      ERR_MEMORY              dynamic memory management failure
 * 
 *   errors returned
 *      OK                     No error
 *      ERR_MEMORY             Dynamic memory management failure
 *      WARN_NO_CMDS           No commands parsed
 *      ERR_TOO_MANY_COMMANDS  too many pipes used
 *   
 *   console messages
 *      CMD_WARN_NO_CMD        print on WARN_NO_CMDS
 *      CMD_ERR_PIPE_LIMIT     print on ERR_TOO_MANY_COMMANDS
 *      CMD_ERR_EXECUTE        print on execution failure of external command
 * 
 *  Standard Library Functions You Might Want To Consider Using (assignment 1+)
 *      malloc(), free(), strlen(), fgets(), strcspn(), printf()
 * 
 *  Standard Library Functions You Might Want To Consider Using (assignment 2+)
 *      fork(), execvp(), exit(), chdir()
 */

void trim_spaces(char *str) {
	while (*str && isspace((unsigned char)*str)) {
		str++;
	}

	size_t len = strlen(str);
	while (len > 0 && isspace((unsigned char)str[len - 1])) {
		str[len - 1] = '\0';
		len--;
	}
}

Built_In_Cmds match_command(const char *input) {
	if(strcmp(input, EXIT_CMD) == 0) {
		return BI_CMD_EXIT;
	}
	else if(strcmp(input, "dragon") == 0) {
		return BI_CMD_DRAGON;
	}
	else if(strcmp(input, "cd") == 0) {
		return BI_CMD_CD;
	}
	else {
		return BI_NOT_BI;
	}
}

Built_In_Cmds exec_built_in_cmd(cmd_buff_t *cmd) {
	if(cmd->argc == 0) {
		return BI_NOT_BI;
	}

	if(strcmp(cmd->argv[0], "cd") == 0) {
		if(cmd->argc == 2) {
			if(chdir(cmd->argv[1]) == -1) {
				perror("cd failed");
			}
		}

		return BI_CMD_CD;
	}

	return BI_NOT_BI;
}

int build_cmd_list(char *cmd_line, command_list_t *clist) {
	//printf("Before trimming the cmd_line in build_cmd_list: '%s'\n", cmd_line);
	trim_spaces(cmd_line);	
	//printf("Trimmed cmd_line: '%s'\n", cmd_line);

	char *cmd_tok_save = NULL;
	char *cmd_tok = strtok_r(cmd_line, PIPE_STRING, &cmd_tok_save);

	int index = 0;

	while(cmd_tok != NULL) {
		trim_spaces(cmd_tok);

		//printf("Command %d: '%s'\n", index, cmd_tok);

		if(index >= CMD_MAX) {
			return ERR_TOO_MANY_COMMANDS;
		}

		cmd_buff_t *cmd = &clist->commands[index];
		cmd->argc = 0;
		cmd->_cmd_buffer = cmd_tok;
		cmd->input_redirect = NULL;
		cmd->output_redirect = NULL;
		
		//printf("Command %d (before args parsing): '%s'\n", index, cmd_tok);

		char *arg_tok_save = NULL;
		char *arg_tok = strtok_r(cmd_tok, " ", &arg_tok_save);

		while(arg_tok != NULL && cmd->argc < CMD_ARGV_MAX) {
			trim_spaces(arg_tok);

			//printf("Command %d: argv[%d]: '%s'\n", index, cmd->argc, arg_tok);

			if(strcmp(arg_tok, "<") == 0) {
				arg_tok = strtok_r(NULL, " ", &arg_tok_save);
				if(arg_tok != NULL) {
					cmd->input_redirect = arg_tok;
				}
			}
			else if(strcmp(arg_tok, ">>") == 0) {
				arg_tok = strtok_r(NULL, " ", &arg_tok_save);
				if(arg_tok != NULL) {
					cmd->output_redirect = arg_tok;
					cmd->output_append = 1;
				}
			}
			else if(strcmp(arg_tok, ">") == 0) {
				arg_tok = strtok_r(NULL, " ", &arg_tok_save);
				if(arg_tok != NULL) {
					cmd->output_redirect = arg_tok;
				}
			}
			else {
				cmd->argv[cmd->argc] = arg_tok;
				cmd->argc++;
			}

			arg_tok = strtok_r(NULL, " ", &arg_tok_save);
		}

		cmd->argv[cmd->argc] = NULL;

		//printf("Command %d finished parsing. argc = %d\n", index, cmd->argc);

		index++;
		cmd_tok = strtok_r(NULL, PIPE_STRING, &cmd_tok_save);
	}

	clist->num = index;
	//printf("Total number of commands: %d\n", clist->num);
	return OK;
}

int exec_cmd(cmd_buff_t *cmd, int cmd_index, command_list_t *clist, int pipefd_in) {
	int pipefd[2];
	if(pipe(pipefd) == -1) {
		perror("pipe");
		exit(EXIT_FAILURE);
	}

	pid_t pid = fork();
	if(pid == -1) {
		perror("fork");
		exit(EXIT_FAILURE);
	}

	if(pid == 0) {
		if(cmd_index > 0) {
			close(pipefd[0]);
		}

		if(cmd_index < clist->num - 1) {
			close(pipefd[0]);
		}

		if(cmd->input_redirect != NULL) {
			int input_fd = open(cmd->input_redirect, O_RDONLY);
			if(input_fd == -1) {
				perror("open");
				exit(EXIT_FAILURE);
			}
			dup2(input_fd, STDIN_FILENO);
			close(input_fd);
		}
		else if(cmd_index > 0) {
			dup2(pipefd_in, STDIN_FILENO);
			close(pipefd_in);
		}

		if(cmd->output_redirect != NULL) {
			int flags = O_WRONLY | O_CREAT;
			if(cmd->output_append) {
				flags |= O_APPEND;
				//printf("OPENING FILE IN APPEND MODE: output_redirect = %s\n", cmd->output_redirect);
			}
			else {
				flags |= O_TRUNC;
				//printf("OPENING FILE IN OVERWRITE MODE: output_redirect = %s\n", cmd->output_redirect);
			}
			int output_fd = open(cmd->output_redirect, flags, 0644);
			if(output_fd == -1) {
				perror("open");
				exit(EXIT_FAILURE);
			}
			//printf("FILE OPENED: output_fd = %d\n", output_fd);
			dup2(output_fd, STDOUT_FILENO);
			close(output_fd);
		}
		else if(cmd_index < clist->num - 1) {
			close(STDOUT_FILENO);
			dup2(pipefd[1], STDOUT_FILENO);
			close(pipefd[1]);
		}

		execvp(cmd->argv[0], cmd->argv);

		perror("execvp");
		exit(EXIT_FAILURE);
	}
	else {
		close(pipefd[1]);

		if(cmd_index < clist->num - 1) {
			int next_pipefd = pipefd[0];
			waitpid(pid, NULL, 0);
			return next_pipefd;
		}
		else {
			waitpid(pid, NULL, 0);
			close(pipefd[0]);
			return -1;
		}
	}
}

int exec_local_cmd_loop() {

	char *cmd_buff = malloc(SH_CMD_MAX * sizeof(char));
	if (cmd_buff == NULL) {
		fprintf(stderr, "Memory allocation failed\n");
		return ERR_MEMORY;
	}

	int rc = 0;

	command_list_t clist;


	while(1) {
		printf("%s", SH_PROMPT);
		if(fgets(cmd_buff, ARG_MAX, stdin) == NULL) {
			printf("\n");
			break;
		}

		//printf("Raw input: '%s'\n", cmd_buff);

		// remove the trailing \n from cmd_buff
		cmd_buff[strcspn(cmd_buff, "\n")] = '\0';

		// IMPLEMENT THE REST OF THE REQUIREMENTS

		trim_spaces(cmd_buff);
		//printf("Trimmed input in exec_local_cmd_loop: '%s'\n", cmd_buff);

		if(strlen(cmd_buff) == 0) {
			printf(CMD_WARN_NO_CMD);
			continue;
		}

		if(strcmp(cmd_buff, EXIT_CMD) == 0) {
			free(cmd_buff);
			exit(OK);
		}

		rc = build_cmd_list(cmd_buff, &clist);
		//printf("RC value: %d\n", rc);

		switch(rc) {
			case OK:
				int pipefd_in = -1;
				for(int i = 0; i < clist.num; i++) {
					if(exec_built_in_cmd(&clist.commands[i]) == BI_NOT_BI) {
						pipefd_in = exec_cmd(&clist.commands[i], i, &clist, pipefd_in);
					}
				}
				break;
			case WARN_NO_CMDS:
				printf(CMD_WARN_NO_CMD);
				break;
			case ERR_TOO_MANY_COMMANDS:
				printf(CMD_ERR_PIPE_LIMIT, CMD_MAX);
				break;
		}
	}

	free(cmd_buff);
	return OK;
}
