const cellTypes = {
  EMPTY: 0,
  PLAYERHEAD: 1,
  PLAYERBODY: 2,
  FOOD: 3,
  BANANA: 4,
  BORDER: 5,
};

const COLLISIONTYPES = {
  NONE: 0,
  FOOD: 1,
  BANANA: 2,
  BORDER: 3,
  SNAKE: 4,
};

function createCell(x, y, type) {
  return {
    x: x,
    y: y,
    type: type,
    pid: 0,
    direction: 0,
    next: null,
  };
}

function getRandomInt(max) {
  return Math.floor(Math.random() * max);
}

function addFood(gameBoard) {
  //Find all the empty spaces on the grid
  let emptySpaces = [];
  for (let row of gameBoard) {
    for (let cell of row) {
      if (cell.type === cellTypes.EMPTY) {
        emptySpaces.push(cell);
      }
    }
  }

  let bananaExists = gameBoard.some((row) =>
    row.some((cell) => cell.type === cellTypes.BANANA),
  );

  //Create food cells randomly
  if (emptySpaces.length > 0) {
    let randomCell = emptySpaces[getRandomInt(emptySpaces.length)];
    randomCell.type = cellTypes.FOOD;
  }

  if (!bananaExists && emptySpaces.length > 0) {
    let randomCell = emptySpaces[getRandomInt(emptySpaces.length)];
    randomCell.type = cellTypes.BANANA;
  }

  return gameBoard;
}

function checkCollisions(newY, newX, gameBoard) {
  // returns a list of players that have collided,
  // and a new gameBoard with the players removed
  if (
    newX < 0 ||
    newX >= gameBoard[0].length ||
    newY < 0 ||
    newY >= gameBoard.length
  ) {
    console.log(`Player has collided with the wall!`);
    return COLLISIONTYPES.BORDER;
  }
  //Handle collision with food
  if (gameBoard[newY][newX].type === cellTypes.FOOD) {
    console.log(`Player has eaten the food!`);
    return COLLISIONTYPES.FOOD;
  }
  if (gameBoard[newY][newX].type === cellTypes.PLAYERHEAD) {
    console.log(`Player has collided with another player!`);
    return COLLISIONTYPES.SNAKE;
  }
  //Handle collision with banana
  if (gameBoard[newY][newX].type == cellTypes.BANANA) {
    console.log("Slip");
    return COLLISIONTYPES.BANANA;
  }
  //Handle collision with border
  if (gameBoard[newY][newX].type == cellTypes.BORDER) {
    console.log("Void");
    return COLLISIONTYPES.BORDER;
  }
  return COLLISIONTYPES.NONE;
}

module.exports = {
  cellTypes,
  COLLISIONTYPES,

  createGameBoard: (width, height) => {
    let gameBoard = [];
    for (var i = 0; i < height; i++) {
      let row = [];
      for (var j = 0; j < width; j++) {
        row.push(createCell(j, i, cellTypes.EMPTY));
      }
      gameBoard.push(row);
    }
    for (var i = 0; i < 10; i++) {
      gameBoard = addFood(gameBoard);
    }
    //console.log(gameBoard);
    return gameBoard;
  },

  addPlayer: (gameBoard, pid) => {
    let placedCell = false;
    while (!placedCell) {
      let x = getRandomInt(gameBoard[0].length);
      let y = getRandomInt(gameBoard.length);

      if (gameBoard[y][x].type === cellTypes.EMPTY) {
        gameBoard[y][x].type = cellTypes.PLAYERHEAD;
        gameBoard[y][x].pid = pid;

        let bodyPlaced = false;

        if (y - 1 >= 0 && gameBoard[y - 1][x].type === cellTypes.EMPTY) {
          //Place body segment
          gameBoard[y - 1][x] = createCell(x, y - 1, cellTypes.PLAYERBODY);
          gameBoard[y - 1][x].pid = pid;
          gameBoard[y][x].next = gameBoard[y - 1][x];
          bodyPlaced = true;
        }
        if (bodyPlaced) {
          placedCell = true;
        } else {
          gameBoard[y][x] = createCell(x, y, cellTypes.EMPTY);
        }
      }
    }
    //console.log(gameBoard);
    return gameBoard;
  },

  moveOneStep: (gameBoard) => {
    //Save board state to allow multiple snake movements
    let updatedBoard = gameBoard.map((row) => row.map((cell) => ({ ...cell })));
    let deadPlayers = [];

    //Loop through board until we find a PLAYERHEAD
    for (var i = 0; i < gameBoard.length; i++) {
      for (var j = 0; j < gameBoard[i].length; j++) {
        let cell = gameBoard[i][j];
        if (cell.type === cellTypes.PLAYERHEAD) {
          let oldX = cell.x;
          let oldY = cell.y;
          let newX = oldX;
          let newY = oldY;

          let tempVar = cell.direction;

          //New position based on direction
          if (cell.direction === 0) {
            //Up
            newY -= 1;
          } else if (cell.direction === 1) {
            //Right
            newX += 1;
          } else if (cell.direction === 2) {
            //Down
            newY += 1;
          } else if (cell.direction === 3) {
            //Left
            newX -= 1;
          }

          let collision = checkCollisions(newY, newX, gameBoard);

          if (
            collision === COLLISIONTYPES.BANANA ||
            collision === COLLISIONTYPES.FOOD
          ) {
            updatedBoard = addFood(updatedBoard);
          }
          if (collision === COLLISIONTYPES.SNAKE) {
            updatedBoard = updatedBoard.map((row) =>
              row.map((c) =>
                c.pid === cell.pid ? createCell(c.x, c.y, cellTypes.EMPTY) : c,
              ),
            );
            deadPlayers.push(cell.pid);
            updatedBoard = updatedBoard.map((row) =>
              row.map((c) =>
                c.pid === gameBoard[newY][newX].pid
                  ? createCell(c.x, c.y, cellTypes.EMPTY)
                  : c,
              ),
            );
            deadPlayers.push(gameBoard[newY][newX].pid);
            continue;
          }
          if (collision > COLLISIONTYPES.FOOD) {
            updatedBoard = updatedBoard.map((row) =>
              row.map((c) =>
                c.pid === cell.pid ? createCell(c.x, c.y, cellTypes.EMPTY) : c,
              ),
            );
            deadPlayers.push(cell.pid);
            continue;
          }
          if (collision === COLLISIONTYPES.FOOD) {
            let newBodySegment = createCell(oldX, oldY, cellTypes.PLAYERBODY);
            newBodySegment.pid = cell.pid;
            newBodySegment.next = cell.next;
            cell.next = newBodySegment;
          }

          //Snake head to new position by updating cell object
          let newHeadCell = createCell(newX, newY, cellTypes.PLAYERHEAD);
          newHeadCell.pid = cell.pid;
          newHeadCell.direction = tempVar;
          newHeadCell.next = cell.next;
          updatedBoard[newY][newX] = newHeadCell;

          //Remove previous head position
          updatedBoard[oldY][oldX] = createCell(oldX, oldY, cellTypes.EMPTY);

          //Move the body segments
          let currentBody = newHeadCell.next;
          while (currentBody) {
            let bodyOldX = currentBody.x;
            let bodyOldY = currentBody.y;
            currentBody.x = oldX;
            currentBody.y = oldY;

            //Update cell body position
            updatedBoard[oldY][oldX] = {
              ...currentBody,
              type: cellTypes.PLAYERBODY,
            };
            oldX = bodyOldX;
            oldY = bodyOldY;
            currentBody = currentBody.next;
          }

          //Remove previous body position if it's no longer part of the snake
          if (oldX !== newHeadCell.x || oldY !== newHeadCell.y) {
            updatedBoard[oldY][oldX] = createCell(oldX, oldY, cellTypes.EMPTY);
          }
        }
      }
    }
    return [updatedBoard, deadPlayers];
  },

  applyBorders: (gameBoard, borderCounter) => {
    let height = gameBoard.length;
    let width = gameBoard[0].length;

    //console.log(borderCounter);

    //Return gameBoard when it is 2x2
    if (borderCounter * 2 >= Math.min(height, width) - 2) {
      return gameBoard;
    }

    //Top
    for (let x = 0; x < width; x++) {
      gameBoard[borderCounter][x].type = cellTypes.BORDER;
      gameBoard[borderCounter][x].pid = 0;
      gameBoard[borderCounter][x].direction = 0;
      gameBoard[borderCounter][x].next = null;
    }

    //Bottom
    for (let x = 0; x < width; x++) {
      gameBoard[height - borderCounter - 1][x].type = cellTypes.BORDER;
      gameBoard[height - borderCounter - 1][x].pid = 0;
      gameBoard[height - borderCounter - 1][x].direction = 0;
      gameBoard[height - borderCounter - 1][x].next = null;
    }

    //Left
    for (let y = 0; y < height; y++) {
      gameBoard[y][borderCounter].type = cellTypes.BORDER;
      gameBoard[y][borderCounter].pid = 0;
      gameBoard[y][borderCounter].direction = 0;
      gameBoard[y][borderCounter].next = null;
    }

    //Right
    for (let y = 0; y < height; y++) {
      gameBoard[y][width - borderCounter - 1].type = cellTypes.BORDER;
      gameBoard[y][width - borderCounter - 1].pid = 0;
      gameBoard[y][width - borderCounter - 1].direction = 0;
      gameBoard[y][width - borderCounter - 1].next = null;
    }

    return gameBoard;
  },

  getPlayerView: (gameBoard, pid) => {
    let updatedBoard = gameBoard.map((row) => row.map((cell) => ({ ...cell })));

    let width = 11;
    let height = 11;

    let head = null;
    for (var i = 0; i < updatedBoard.length; i++) {
      for (var j = 0; j < updatedBoard[i].length; j++) {
        let cell = updatedBoard[i][j];
        if (cell.type === cellTypes.PLAYERHEAD && cell.pid === pid) {
          console.log("MY Player head found at: ", cell.x, cell.y);
          head = cell;
        }
      }
    }

    if (head === null) {
      return [];
    }

    let playerView = [];

    for (var i = 0; i <= height; i++) {
      let row = [];
      for (var j = 0; j <= width; j++) {
        let x = head.x + j - Math.floor(width / 2);
        let y = head.y + i - Math.floor(height / 2);

        if (
          x < 0 ||
          x >= updatedBoard[0].length ||
          y < 0 ||
          y >= updatedBoard.length
        ) {
          row.push(cellTypes.BORDER);
        } else {
          row.push(updatedBoard[y][x]);
        }
      }
      playerView.push(row);
    }

    return playerView;
  },
};
