const { WebSocketServer } = require("ws");
const gameModule = require("../game/game.js");

class WebSocketModel {
  createInterval(game) {
    setInterval(() => {
      if (!this.games[game].started) return;
      let deadPlayers = [];

      [this.games[game].gameBoard, deadPlayers] = gameModule.moveOneStep(
        this.games[game].gameBoard,
      );

      if (!this.games[game].elapsedTime) {
        this.games[game].elapsedTime = 0;
      }
      this.games[game].elapsedTime++;

      //Every 10 seconds, generate borders
      if (this.games[game].elapsedTime % 10 === 0) {
        this.games[game].gameBoard = gameModule.applyBorders(
          this.games[game].gameBoard,
          this.games[game].borderCounter,
        );
        this.games[game].borderCounter++;
      }

      if (deadPlayers.length > 0) {
        for (let conn of this.games[game].players) {
          for (let deadPlayer of deadPlayers) {
            if (conn.pid === deadPlayer) {
              conn.spectating = true;
            }
          }
          conn.connection.send(
            JSON.stringify({
              type: "deadPlayers",
              data: deadPlayers,
            }),
          );
        }
      }

      for (let conn of this.games[game].players) {
        let playerView = null;
        if (conn.spectating) {
          playerView = gameModule.getPlayerView(
            this.games[game].gameBoard,
            conn.spectatingPlayer,
          );
        } else {
          playerView = gameModule.getPlayerView(
            this.games[game].gameBoard,
            conn.pid,
          );
        }
        conn.connection.send(
          JSON.stringify({
            type: "gameBoard",
            data: playerView,
          }),
        );
      }
    }, 500);
  }
  constructor() {
    this.connections = [];
    this.games = {};
    this.sockserver = new WebSocketServer({ port: 3001 });
    this.createGame("game1", "public");
    this.createGame("game2", "public");
    this.createGame("game3", "public");
    this.onConnection();
  }

  hasGame(gameId) {
    return this.games.hasOwnProperty(gameId);
  }

  getGameIds() {
    let publicGames = [];
    publicGames = Object.keys(this.games).filter(
      (game) => this.games[game].type === "public",
    );
    return publicGames;
  }

  createGame(gameId, gameType) {
    this.games[gameId] = {
      gameBoard: gameModule.createGameBoard(100, 100),
      players: [],
      type: gameType,
      started: false,
      readyPlayers: 0,
      borderCounter: 0,
      numPlayers: 0,
    };
    this.createInterval(gameId);
  }

  moveHandler(connection, message) {
    console.log("Received move: " + message.data);
    let movement = message.data;
    let pid = message.pid;
    let gameBoard = this.games[connection.protocol].gameBoard;
    console.log("pid: ", pid);

    for (var i = 0; i < gameBoard.length; i++) {
      for (var j = 0; j < gameBoard[0].length; j++) {
        if (gameBoard[i][j].type === 1 && gameBoard[i][j].pid === pid) {
          console.log("Updating direction to: " + movement);
          switch (movement) {
            case "up":
              gameBoard[i][j].direction = 0;
              break;
            case "right":
              gameBoard[i][j].direction = 1;
              cell.direction = 1;
              break;
            case "down":
              gameBoard[i][j].direction = 2;
              break;
            case "left":
              gameBoard[i][j].direction = 3;
              break;
          }
          console.log("New direction set to: " + gameBoard[i][j].direction);
          return;
        }
      }
    }
  }

  onConnection() {
    this.sockserver.on("connection", (connection) => {
      console.log("New client connected!");

      console.log(connection.protocol);

      let roomId = connection.protocol;

      if (
        !this.games[roomId].started &&
        this.games[roomId].players.length < 10
      ) {
        this.games[roomId].players.push({
          connection,
          pid: this.games[roomId].players.length + 1,
          spectating: false,
          spectatingPlayer: 1,
        });
        this.games[roomId].numPlayers = this.games[roomId].numPlayers + 1;
        this.games[roomId].gameBoard = gameModule.addPlayer(
          this.games[roomId].gameBoard,
          this.games[roomId].players.length,
        );
      } else {
        this.games[roomId].players.push({
          connection,
          pid: this.games[roomId].players.length + 1,
          spectating: true,
          spectatingPlayer: 1,
        });
      }

      connection.send(
        JSON.stringify({
          type: "newUser",
          data: this.games[roomId].players.length,
        }),
      );

      connection.on("close", () => {
        // Need to fix this
        this.games[roomId].players = this.games[roomId].players.filter(
          (curr) => curr !== this.games[roomId].players,
        );
        console.log("Client has disconnected!");
      });

      connection.on("message", (event) => {
        try {
          let message = JSON.parse(event);
          console.log("Received message: " + message);
          //All messages are expected to have a type
          if (message.type === "move") {
            console.log("Received move: " + message.data);
            this.moveHandler(connection, message);
          } else if (message.type === "chat") {
            console.log("Received chat: " + message.data);
            for (let conn of this.games[roomId].players) {
              conn.connection.send(
                JSON.stringify({
                  type: "chat",
                  data: message.data,
                  pid: message.pid,
                }),
              );
            }
          } else if (message.type === "join") {
            console.log("Received join: " + message.data);
            for (let conn of this.games[roomId].players) {
              conn.connection.send(
                JSON.stringify({
                  type: "playerJoined",
                  data: message.data,
                  numPlayers: this.games[roomId].players.length,
                  pid: message.pid,
                }),
              );
            }
          } else if (message.type === "leave") {
            console.log("Received leave: " + message.data);
          } else if (message.type === "spectate") {
            console.log("Received spectate: " + message.data);
            for (let conn of this.games[roomId].players) {
              if (conn.connection === connection) {
                let player = conn.spectatingPlayer;
                if (player === this.games[roomId].numPlayers) {
                  conn.spectatingPlayer = 1;
                } else {
                  conn.spectatingPlayer++;
                }
              }
            }
          } else if (message.type === "start") {
            console.log("Received start: " + message.data);
            for (let conn of this.games[roomId].players) {
              if (conn.connection !== connection)
                conn.connection.send(
                  JSON.stringify({ type: "playerReady", data: message.data }),
                );
            }
            this.games[roomId].readyPlayers++;
            if (
              this.games[roomId].players.length ===
              this.games[roomId].readyPlayers
            ) {
              for (let conn of this.games[roomId].players) {
                conn.connection.send(
                  JSON.stringify({ type: "start", data: message.data }),
                );
              }
              this.games[roomId].started = true;
            }
          } else {
            console.log("Received: " + message);
          }
        } catch (e) {
          console.log("Error parsing JSON: " + e.message);
        }
      });

      connection.onerror = function () {
        console.log("websocket error");
      };
    });
  }
}

module.exports = new WebSocketModel();
