const { WebSocketServer } = require("ws");
const gameModule = require("../game/game.js");

class WebSocketModel {
	createInterval(game) {
	  setInterval(() => {
		if (!this.games[game].started) return;
		let deadPlayers = [];

		// Get the current time
		const currentTime = Date.now();

		// Move all players based on their speed
		for (let player of this.games[game].players) {
		  if (player.spectating) continue;

		  // Initialize lastMoveTime if it doesn't exist
		  if (!player.lastMoveTime) {
			player.lastMoveTime = currentTime;
		  }

		  // Move the player if enough time has passed since their last move
		  if (currentTime - player.lastMoveTime >= player.speed) {
			// Move the player
			[this.games[game].gameBoard, deadPlayers] = gameModule.moveOneStep(
			  this.games[game].gameBoard,
			);

			// Update the player's last move time
			player.lastMoveTime = currentTime;

			// Check if the player collided with the energy cell
			let playerCollision = this.checkPlayerCollision(player.pid, this.games[game].gameBoard);
			if (playerCollision === gameModule.COLLISIONTYPES.ENERGY) {
			  console.log(`Player ${player.pid} has eaten an energy cell!`);

			  // Boost the speed for 5 seconds
			  player.speed = 250; // Double speed (250ms)
			  setTimeout(() => {
				player.speed = 500; // Reset speed after 5 seconds
			  }, 5000);
			}
		  }
		}

		if (deadPlayers.length > 0) {
		  this.handleDeadPlayers(game, deadPlayers);
		}

		this.updatePlayersView(game);
	  }, 100); // Check every 100ms for more fine-grained control
	}

  constructor() {
    this.connections = [];
    this.games = {};
    this.sockserver = new WebSocketServer({ port: 3001 });
    this.createGame("game1", "public");
    this.createGame("game2", "public");
    this.createGame("game3", "public");
    this.onConnection();
  }

  hasGame(gameId) {
    return this.games.hasOwnProperty(gameId);
  }

  getGameIds() {
    let publicGames = [];
    publicGames = Object.keys(this.games).filter(
      (game) => this.games[game].type === "public",
    );
    return publicGames;
  }

  createGame(gameId, gameType) {
    this.games[gameId] = {
      gameBoard: gameModule.createGameBoard(100, 100),
      players: [],
      type: gameType,
      started: false,
      readyPlayers: 0,
      borderCounter: 0,
      numPlayers: 0,
    };
    this.createInterval(gameId);
  }

  resetGame(gameId) {
    this.games[gameId].gameBoard = gameModule.createGameBoard(100, 100);
    this.games[gameId].players.forEach((player) => {
      player.spectating = false;
      player.speed = 500; // Reset speed to default
    });
    this.games[gameId].started = false;
    this.games[gameId].readyPlayers = 0;
    this.games[gameId].borderCounter = 0;
    this.games[gameId].numPlayers = 0;
    this.games[gameId].elapsedTime = 0;
  }

  moveHandler(connection, message) {
    console.log("Received move: " + message.data);
    let movement = message.data;
    let pid = message.pid;
    let gameBoard = this.games[connection.protocol].gameBoard;
    console.log("pid: ", pid);

    for (var i = 0; i < gameBoard.length; i++) {
      for (var j = 0; j < gameBoard[0].length; j++) {
        if (gameBoard[i][j].type === 1 && gameBoard[i][j].pid === pid) {
          console.log("Updating direction to: " + movement);
		  
		  let currentDirection = gameBoard[i][j].direction;
          let newDirection = currentDirection;
		  
          switch (movement) {
            case "up":
              if (currentDirection !== 2) newDirection = 0; // Cannot move up if going down
              break;
            case "right":
              if (currentDirection !== 3) newDirection = 1; // Cannot move right if going left
              break;
            case "down":
              if (currentDirection !== 0) newDirection = 2; // Cannot move down if going up
              break;
            case "left":
              if (currentDirection !== 1) newDirection = 3; // Cannot move left if going right
              break;
          }

          if (newDirection !== currentDirection) {
            gameBoard[i][j].direction = newDirection;
          }
          return;
        }
      }
    }
  }

  checkPlayerCollision(pid, gameBoard) {
    // Check if the player's head has collided with any special cells
    for (let row of gameBoard) {
      for (let cell of row) {
        if (cell.pid === pid && cell.type === gameModule.cellTypes.PLAYERHEAD) {
          if (cell.next && cell.next.type === gameModule.cellTypes.ENERGY) {
            return gameModule.COLLISIONTYPES.ENERGY;
          }
        }
      }
    }
    return gameModule.COLLISIONTYPES.NONE;
  }

  handleDeadPlayers(game, deadPlayers) {
    for (let conn of this.games[game].players) {
      for (let deadPlayer of deadPlayers) {
        if (conn.pid === deadPlayer) {
          conn.spectating = true;
        }
      }
      conn.connection.send(
        JSON.stringify({
          type: "deadPlayers",
          data: deadPlayers,
        }),
      );
    }

    const alivePlayers = this.games[game].players.filter(
      (player) => !player.spectating
    );

    if (alivePlayers.length === 1) {
      const winner = alivePlayers[0].pid;

      for (let conn of this.games[game].players) {
        conn.connection.send(
          JSON.stringify({
            type: "restart",
            pid: winner,
          }),
        );
      }

      this.resetGame(game);
    }
  }

  updatePlayersView(game) {
    for (let conn of this.games[game].players) {
      let playerView = null;
      if (conn.spectating) {
        playerView = gameModule.getPlayerView(
          this.games[game].gameBoard,
          conn.spectatingPlayer,
        );
      } else {
        playerView = gameModule.getPlayerView(
          this.games[game].gameBoard,
          conn.pid,
        );
      }
      conn.connection.send(
        JSON.stringify({
          type: "gameBoard",
          data: playerView.view,
          headPosition: playerView.headPosition,
          bodySize: playerView.bodySize,
        }),
      );
    }
  }

  onConnection() {
    this.sockserver.on("connection", (connection) => {
      console.log("New client connected!");

      let roomId = connection.protocol;

      if (
        !this.games[roomId].started &&
        this.games[roomId].players.length < 10
      ) {
        this.games[roomId].players.push({
          connection,
          pid: this.games[roomId].players.length + 1,
          spectating: false,
          spectatingPlayer: 1,
          speed: 500, // Default speed
        });
        this.games[roomId].numPlayers++;
        this.games[roomId].gameBoard = gameModule.addPlayer(
          this.games[roomId].gameBoard,
          this.games[roomId].players.length,
        );
      } else {
        this.games[roomId].players.push({
          connection,
          pid: this.games[roomId].players.length + 1,
          spectating: true,
          spectatingPlayer: 1,
          speed: 500, // Default speed even for spectators
        });
      }

      connection.send(
        JSON.stringify({
          type: "newUser",
          data: this.games[roomId].players.length,
        }),
      );

      connection.on("close", () => {
        this.games[roomId].players = this.games[roomId].players.filter(
          (curr) => curr !== this.games[roomId].players,
        );
        console.log("Client has disconnected!");
      });

      connection.on("message", (event) => {
        try {
          let message = JSON.parse(event);
          if (message.type === "move") {
            this.moveHandler(connection, message);
          } else if (message.type === "chat") {
            for (let conn of this.games[roomId].players) {
              conn.connection.send(
                JSON.stringify({
                  type: "chat",
                  data: message.data,
                  pid: message.pid,
                }),
              );
            }
          } else if (message.type === "join") {
            for (let conn of this.games[roomId].players) {
              conn.connection.send(
                JSON.stringify({
                  type: "playerJoined",
                  data: message.data,
                  numPlayers: this.games[roomId].players.length,
                  pid: message.pid,
                }),
              );
            }
          } else if (message.type === "spectate") {
            for (let conn of this.games[roomId].players) {
              if (conn.connection === connection) {
                let player = conn.spectatingPlayer;
                if (player === this.games[roomId].numPlayers) {
                  conn.spectatingPlayer = 1;
                } else {
                  conn.spectatingPlayer++;
                }
              }
            }
          } else if (message.type === "start") {
            for (let conn of this.games[roomId].players) {
              if (conn.connection !== connection)
                conn.connection.send(
                  JSON.stringify({ type: "playerReady", data: message.data }),
                );
            }
            this.games[roomId].readyPlayers++;
            if (
              this.games[roomId].players.length ===
              this.games[roomId].readyPlayers
            ) {
              for (let conn of this.games[roomId].players) {
                conn.connection.send(
                  JSON.stringify({ type: "start", data: message.data }),
                );
              }
              this.games[roomId].started = true;
            }
          }
        } catch (e) {
          console.log("Error parsing JSON: " + e.message);
        }
      });
    });
  }
}

module.exports = new WebSocketModel();
